#!/usr/bin/env python3
# -*- coding: utf-8 -*-
from __future__ import annotations
import argparse, csv, json, os, sys, hashlib, datetime, platform

CSV_NAME = "AR_V1_sim_registry.csv"
JSON_NAME = "AR_V1_ai_pack_v1.0.0.json"

def sha256_of_file(path: str) -> str:
    h = hashlib.sha256()
    with open(path, "rb") as f:
        for chunk in iter(lambda: f.read(1<<20), b""):
            h.update(chunk)
    return h.hexdigest()

def sha256_of_text(text: str) -> str:
    return hashlib.sha256(text.encode("utf-8")).hexdigest()

def detect_run_env() -> str:
    py = f"Python {sys.version_info.major}.{sys.version_info.minor}.{sys.version_info.micro}"
    os_str = f"{platform.system()} {platform.release()}"
    extras = []
    for mod, label in (("numpy","NumPy"),("pytest","pytest")):
        try:
            m = __import__(mod)
            ver = getattr(m, "__version__", "?")
            extras.append(f"{label} {ver}")
        except Exception:
            pass
    return f"{os_str}; {py}; " + (", ".join(extras) if extras else "no extra libs detected")

def ensure_outputs(out_dir: str):
    os.makedirs(out_dir, exist_ok=True)
    csv_path = os.path.join(out_dir, CSV_NAME)
    json_path = os.path.join(out_dir, JSON_NAME)
    if not os.path.exists(csv_path):
        with open(csv_path, "w", encoding="utf-8", newline="") as f:
            writer = csv.writer(f)
            writer.writerow([
                "sim_id","repo_url","commit_hash","seed",
                "config_ref","config_hash","run_env","metrics",
                "result_summary","date_ran","status","outputs_hashes"
            ])
    if not os.path.exists(json_path):
        with open(json_path, "w", encoding="utf-8") as f:
            json.dump({"sims": []}, f, indent=2)
    return csv_path, json_path

def parse_metrics(metrics_str: str) -> dict[str,str]:
    res = {}
    if not metrics_str: return res
    for part in (p.strip() for p in metrics_str.split(",")):
        if not part: continue
        if "=" in part:
            k, v = part.split("=", 1)
            res[k.strip()] = v.strip()
        else:
            res[part] = ""
    return res

def coerce_jsony(s: str):
    low = s.lower()
    if low in ("true","false"): return (low == "true")
    try:
        if "." in s or "e" in low: return float(s)
        return int(s)
    except Exception:
        return s

# ---- Acceptance rule for LIB:V1-ops-core-001 ----
def accept_lib_ops_core(metric_map: dict[str,str]) -> bool:
    ci  = str(metric_map.get("ci_green","")).lower() == "true"
    prim = str(metric_map.get("primitives_pass", metric_map.get("tests_pass",""))).lower() == "true"
    if "coverage" in metric_map:
        try:
            cov = float(metric_map["coverage"])
            cov_min = float(metric_map.get("coverage_min", "0"))
            cov_ok = cov >= cov_min
        except Exception:
            cov_ok = True
    else:
        cov_ok = True
    return ci and prim and cov_ok

ACCEPT_FUN = { "LIB:V1-ops-core-001": accept_lib_ops_core }

def auto_status(sim_id: str, metric_map: dict[str,str]) -> str:
    fn = ACCEPT_FUN.get(sim_id)
    return "VERIFIED" if (fn and fn(metric_map)) else "HOLDING"

def compute_config_hash(args) -> str:
    if args.config_type == "file":
        if not args.config_ref or not os.path.exists(args.config_ref):
            raise FileNotFoundError(f"--config-ref path not found: {args.config_ref}")
        return sha256_of_file(args.config_ref)
    if args.config_type == "manifest":
        if not args.config_ref or not os.path.exists(args.config_ref):
            raise FileNotFoundError(f"--config-ref manifest path not found: {args.config_ref}")
        with open(args.config_ref, "r", encoding="utf-8") as f:
            return sha256_of_text(f.read())
    if args.config_type == "cli":
        if not args.cli_string:
            raise ValueError("--cli-string required when --config-type=cli")
        return sha256_of_text(args.cli_string)
    if args.config_type == "none":
        return "NA"
    raise ValueError(f"Unknown --config-type: {args.config_type}")

def main():
    ap = argparse.ArgumentParser(description="AR provenance recorder (LIB only)")
    sub = ap.add_subparsers(dest="cmd", required=True)

    p_init = sub.add_parser("init"); p_init.add_argument("--out-dir", default=".")
    p_add = sub.add_parser("add")
    p_add.add_argument("--out-dir", default=".")
    p_add.add_argument("--sim-id", required=True)
    p_add.add_argument("--repo-url", required=True)
    p_add.add_argument("--commit-hash", required=True)
    p_add.add_argument("--seed", default="NA")
    p_add.add_argument("--config-type", choices=["file","manifest","cli","none"], required=True)
    p_add.add_argument("--config-ref")
    p_add.add_argument("--cli-string")
    p_add.add_argument("--run-env", default=None)
    p_add.add_argument("--metrics", default="")
    p_add.add_argument("--result-summary", required=True)
    p_add.add_argument("--date-ran", default=None)
    p_add.add_argument("--status", choices=["VERIFIED","HOLDING"], default=None)
    p_add.add_argument("--auto-verify", action="store_true")
    p_add.add_argument("--outputs", default="")

    args = ap.parse_args()

    if args.cmd == "init":
        csv_path, json_path = ensure_outputs(args.out_dir)
        print(f"Initialized:\n  {csv_path}\n  {json_path}")
        return

    if args.cmd == "add":
        csv_path, json_path = ensure_outputs(args.out_dir)
        run_env = args.run_env or detect_run_env()
        date_ran = args.date_ran or datetime.date.today().isoformat()

        try:
            config_hash = compute_config_hash(args)
        except Exception as e:
            print(f"[error] computing config_hash: {e}", file=sys.stderr)
            sys.exit(2)

        metric_map = parse_metrics(args.metrics)
        status = args.status or ("VERIFIED" if (args.auto-verify and auto_status(args.sim_id, metric_map)=="VERIFIED") else "HOLDING")

        outputs_hashes = {}
        if args.outputs:
            for raw in args.outputs.split(";"):
                p = raw.strip()
                if not p: continue
                if os.path.exists(p):
                    try:
                        outputs_hashes[p] = sha256_of_file(p)
                    except Exception as e:
                        outputs_hashes[p] = f"error:{e}"
                else:
                    outputs_hashes[p] = "missing"

        row = [
            args.sim_id, args.repo_url, args.commit_hash, args.seed,
            args.config_ref or "NA", config_hash, run_env,
            args.metrics, args.result_summary, date_ran, status,
            json.dumps(outputs_hashes, ensure_ascii=False),
        ]
        with open(csv_path, "a", encoding="utf-8", newline="") as f:
            csv.writer(f).writerow(row)

        with open(json_path, "r", encoding="utf-8") as f:
            pack = json.load(f)
        entry = {
            "sim_id": args.sim_id, "repo_url": args.repo_url, "commit_hash": args.commit_hash,
            "seed": args.seed, "config_ref": args.config_ref or "NA", "config_hash": config_hash,
            "run_env": run_env, "metrics": {k: coerce_jsony(v) for k,v in metric_map.items()},
            "result_summary": args.result_summary, "date_ran": date_ran, "status": status,
            "outputs_hashes": outputs_hashes,
        }
        pack.setdefault("sims", []).append(entry)
        with open(json_path, "w", encoding="utf-8") as f:
            json.dump(pack, f, indent=2, ensure_ascii=False)

        print(f"Added entry for {args.sim_id} ->\n  CSV: {csv_path}\n  JSON: {json_path}")

if __name__ == "__main__":
    main()
